/**
* \file: am_api_dispatcher.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <sys/epoll.h>
#include <sys/poll.h>
#include <errno.h>
#include <sys/eventfd.h>
#include <unistd.h>

#include "am_api_dispatcher.h"
#include "am_api_events.h"

#include "ipc/message_recvr.h"
#include "utils/logger.h"

//------------------------------------- private attributes ------------------------------------------------------------
typedef enum internal_event_type_t
{
	_NOEVENT,
	ESTABLISH_CONNECTION_SUCCESS,
	ESTABLISH_CONNECTION_FAILURE,
	CONNECTION_LOST
} internal_event_type_t;

static int main_poll_fd=-1;
static int internal_event_fd=-1;

static internal_event_type_t internal_event_type=_NOEVENT;

//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member declaration ----------------------------------------------------
static error_code_t am_api_dispatcher_register_internal_event_fd(void);
static void am_api_dispatcher_on_internal_event(uint32_t events);
static void am_api_dispatcher_send_internal_event(internal_event_type_t type);
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- public member definition ------------------------------------------------------
error_code_t am_api_dispatcher_init(void)
{
	am_api_dispatcher_deinit();

	main_poll_fd=epoll_create1(0);
	if (main_poll_fd<0)
	{
		logger_log_error("AUTOMOUNTER_API - Failed to create an epoll file descriptor in the shared library.");
		return RESULT_NORESOURCE;
	}

	return am_api_dispatcher_register_internal_event_fd();
}

void am_api_dispatcher_deinit(void)
{
	if (main_poll_fd!=-1)
		close(main_poll_fd);
	if (internal_event_fd!=-1)
		close(internal_event_fd);
}

void am_api_dispatcher_signal_establish_connection_success(void)
{
	am_api_dispatcher_send_internal_event(ESTABLISH_CONNECTION_SUCCESS);
}

void am_api_dispatcher_signal_connection_lost(void)
{
	am_api_dispatcher_send_internal_event(CONNECTION_LOST);
}

void am_api_dispatcher_signal_establish_connection_failure(void)
{
	am_api_dispatcher_send_internal_event(ESTABLISH_CONNECTION_FAILURE);
}

int am_api_dispatcher_get_pollfd(void)
{
	return main_poll_fd;
}

void am_api_dispatcher_dispatch_event(void)
{
	struct epoll_event rcv_event;
	int result;

	//Timeout set to 0 here because we don't want to wait, we are called because our fd released another poll.
	// We are calling epoll_wait here just to get the source of our event (internal_fd, socket_fd, or inotify_fd)
	result=epoll_wait(main_poll_fd,&rcv_event,1,0);
	if (result<=0)
		return;

	if (rcv_event.data.ptr!=NULL)
	{
		event_handler_t handler;
		logger_log_debug("AUTOMOUNTER_API - Received a valid event. Going to dispatch it now.'");
		handler=(event_handler_t)rcv_event.data.ptr;
		handler(rcv_event.events);
	}
}

error_code_t am_api_dispatcher_register_event_fd(int fd, event_handler_t handler, uint32_t event_mask)
{
	struct epoll_event event;

	event.events=event_mask;
	event.data.fd=fd;
	event.data.ptr=handler;

	if (epoll_ctl(main_poll_fd,EPOLL_CTL_ADD,fd,&event)==-1)
	{
		logger_log_error("AUTOMOUNTER_API - Failed to add the given file descriptor to the epoll file descriptor.");
		return RESULT_NORESOURCE;
	}

	return RESULT_OK;
}

error_code_t am_api_dispatcher_change_eventfd_eventmask(int fd, event_handler_t handler, uint32_t event_mask)
{
	struct epoll_event event;

	event.events=event_mask;
	event.data.fd=fd;
	event.data.ptr=handler;

	if (epoll_ctl(main_poll_fd,EPOLL_CTL_MOD,fd,&event)==-1)
	{
		if (errno==ENOENT)
			logger_log_error("AUTOMOUNTER_API - Given poll file descriptor is not registered with the pollfd.");
		else
			logger_log_error("AUTOMOUNTER_API - Failed to change the event mask of given poll file descriptor.");
		return RESULT_INVALID;
	}

	return RESULT_OK;
}

void am_api_dispatcher_deregister_event_fd(int fd)
{
	if (epoll_ctl(main_poll_fd,EPOLL_CTL_DEL,fd,NULL)==-1)
		logger_log_error("AUTOMOUNTER_API - Failed to remove the given file descriptor from the epoll file descriptor.");
}

void am_api_dispatcher_on_message_received(message_buffer_t *recvr_buffer)
{
	message_type_t msg_type;
	msg_type=message_recvr_get_msg_type(recvr_buffer);

	//check if we understand the message
	if (msg_type <= __NO_MSG || msg_type >= __LAST_MSG_MARKER)
	{
		logger_log_error("AUTOMOUNTER_API - Message of unknown type received. Probably the shared library has been"
				" compiled with a different version than the automounter has been compiled with. It cannot be"
				"guaranteed that this is running. Please check and change this. For now I'm ignoring the message.");
		return;
	}

	//at the moment, all messages received are resulting directly in events sent to the application. This might change
	//later. There might be internal control messages meant for the library only. Thats why we delegate the "message
	//received events" through this module first. Later, this function can be extended to route the event to other modules
	//depending on the message type.
	am_api_events_on_application_event_message_received(recvr_buffer);
}
//---------------------------------------------------------------------------------------------------------------------

//------------------------------------- private member definition -----------------------------------------------------
static error_code_t am_api_dispatcher_register_internal_event_fd(void)
{
	internal_event_fd=eventfd(0,EFD_NONBLOCK);
	if (internal_event_fd<0)
	{
		logger_log_error("AUTOMOUNTER_API - Failed to create an eventfd file descriptor in the shared library.");
		return RESULT_NORESOURCE;
	}

	return am_api_dispatcher_register_event_fd(internal_event_fd, am_api_dispatcher_on_internal_event,EPOLLIN);
}

static void am_api_dispatcher_on_internal_event(uint32_t events)
{
	uint64_t signal_data;

	//we registered only for EPOLLIN
	(void)events;

	if ((size_t)read(internal_event_fd, &signal_data,sizeof(uint64_t))!=sizeof(uint64_t))
	{
		logger_log_debug("AUTOMOUNTER_API - The internal event fd has been released but I was unable"
				" to read out the uint64 from it.");
		return;
	}

	if (internal_event_type==ESTABLISH_CONNECTION_SUCCESS)
		am_api_events_signal_establish_connection_success();
	else if (internal_event_type==ESTABLISH_CONNECTION_FAILURE)
		am_api_events_signal_establish_connection_failure();
	else if (internal_event_type==CONNECTION_LOST)
		am_api_events_signal_connection_lost();
}

static void am_api_dispatcher_send_internal_event(internal_event_type_t type)
{
	uint64_t signal_data=1;
	internal_event_type=type;

	if ((size_t)write(internal_event_fd, &signal_data, sizeof(uint64_t))!=sizeof(uint64_t))
		logger_log_error("AUTOMOUNTER_API: It seems we have some troubles with the eventfd sending something.");
}
//---------------------------------------------------------------------------------------------------------------------
